# Phase/Status Management Architecture

## Phase API Design (Frontend-Backend Contract)

**Frontend Role:** Phase-agnostic UI
- Creates records with `current_phase: null`
- Displays current phase name (read-only)
- Calls generic endpoints without knowing phase names

**Backend Role:** Phase owner
- Defines phase order and dependencies
- Executes phases based on current state
- Manages all state transitions

**API Endpoints:**
- `POST /api/custom/links/{id}/phase/advance` - Execute current phase, advance on success
- `POST /api/custom/links/{id}/phase/rerun` - Rollback to last completed phase, execute again

**State Ownership:**
- Only Orchestrator modifies `current_phase` and `completed_phases`
- Frontend never writes phase names, only triggers progression

## Overview

This document describes the abstract architecture of the phase/status management system. It explains how processing stages are tracked, validated, and transitioned without reference to specific implementations.

## Core Components

### 1. State Container (Database Record)
- Tracks overall status (pending/processing/completed/failed)
- Maintains current active phase
- Stores list of completed phases
- Holds phase-specific results as structured data

### 2. Phase Orchestrator (Central Manager)
- Defines phase ordering and dependencies
- Validates if a phase can execute
- Updates state after phase completion
- Enforces sequential execution rules

### 3. Phase Executors (Worker Modules)
- Perform actual work for each phase
- Read inputs from state container
- Call orchestrator when finished
- Do not directly modify global state

### 4. API Layer (Entry Points)
- Receives execution requests
- Delegates to executors
- Handles errors and user responses

## State Tracking

### Persistent State
- Array of completed phase names
- Current active phase pointer
- Key-value store for phase outputs
- Global status flag

### Validation State
- File existence checks
- Dependency completion checks
- Execution eligibility status

## Transition Mechanism

### 1. Request Reception
- API receives phase execution request
- Extracts job identifier and target phase

### 2. Pre-Execution Validation
- Orchestrator checks dependency graph
- Verifies all prerequisite phases completed
- Confirms required artifacts exist
- Returns go/no-go decision

### 3. Execution
- Executor performs work
- Reads configuration from state container
- Produces artifacts and results
- Does not modify state during execution

### 4. State Update (Critical)
- Executor calls orchestrator's update method
- Orchestrator atomically updates:
  - Adds phase to completed list
  - Stores results in data store
  - Advances current phase pointer
  - Updates global status if final phase

### 5. Next Phase Determination
- Orchestrator looks up phase sequence
- Sets pointer to next phase in chain
- If last phase: marks job as complete

## Responsibility Model

### Orchestrator (Single Authority)
- ONLY component that modifies phase progression state
- Owns the dependency graph
- Validates execution eligibility
- Manages state transitions

### Executors (Workers)
- Perform domain-specific work
- Report completion to orchestrator
- Never modify state directly
- Stateless operation

### API Layer (Coordinator)
- Receives user intent
- Invokes appropriate executor
- Catches errors
- May set failure status on exceptions

### Database (State Persistence)
- Passive storage
- Provides transaction safety
- Ensures state consistency

## Flow Pattern

```
User Request → API validates job exists
            ↓
API calls Orchestrator.can_run_phase(target)
            ↓
Orchestrator checks: dependencies + files
            ↓
If valid: API invokes Executor
            ↓
Executor does work
            ↓
Executor calls Orchestrator.update_state(completed, results)
            ↓
Orchestrator updates: [completed list] + [current pointer] + [results]
            ↓
Response to user
```

## Key Principles

### Single Source of Truth
Only orchestrator modifies progression state

### Fail-Safe
Phase cannot execute unless all dependencies satisfied

### Idempotent Validation
Can check "can execute?" without side effects

### Sequential Enforcement
Linear dependency chain prevents chaos

### Atomic Transitions
State updates happen in single transaction

### Artifact Validation
File/data existence checked before execution allowed

## State Transitions

### Job Lifecycle
```
Created → Pending → [Phase 1] → [Phase 2] → ... → [Phase N] → Completed
                         ↓
                      Failed (at any point)
```

### Phase Progression
```
Phase Completed → Orchestrator Called → State Updated → Pointer Advanced → Next Phase Ready
```

### Validation Gates
```
Execution Request → Dependency Check → File Check → Permission Granted/Denied
```

## Error Handling

### Validation Failure
- Pre-execution checks fail
- API returns error to user
- No state modification occurs

### Execution Failure
- Executor encounters error
- Does not call orchestrator
- API may mark job as failed
- State remains at last successful phase

### Recovery
- User can retry failed phase
- Validation ensures prerequisites still met
- Can resume from last successful state
