package fetcher

import (
	"clio/internal/processor"

	fetchedfilepb "clio/internal/fetched_file/pocketbase"
)

// Fetcher defines the interface for all content fetchers.
type Fetcher interface {
	// Name returns the fetcher identifier (e.g., "direct_access", "youtube_downloader")
	Name() string

	// Phases returns the phase definitions for this fetcher, in execution order.
	// Phase names should be simple (e.g., "fetch", "download", "extract")
	// They will be namespaced as "fetch:{Name()}:{phase}"
	Phases() []PhaseDefinition

	// GetExecutor returns the executor function for a specific phase.
	// phaseName is the simple name (e.g., "fetch"), not the full namespaced name.
	// store is the fetched_file store for saving files.
	GetExecutor(phaseName string, store *fetchedfilepb.Store) (processor.ExecutorFunc, error)
}

// PhaseDefinition describes a single phase within a fetcher.
type PhaseDefinition struct {
	Name         string   // Simple name (e.g., "fetch", "download")
	Dependencies []string // Simple names of phases this depends on (within same fetcher)
}

// FetchManifest is what each phase returns to describe what was saved.
type FetchManifest struct {
	Files    []FileInfo     // Files created/saved
	Metadata map[string]any // Any metadata extracted
	Summary  string         // Human-readable summary
}

// FileInfo describes a saved file.
type FileInfo struct {
	Path string // File path or storage key
	Type string // MIME type or "html", "video", "image", etc.
	Size int64  // File size in bytes
}
