package pocketbase

import (
	"net/url"
	"strings"

	"clio/internal/fetcher"
	fetchedfilepb "clio/internal/fetched_file/pocketbase"
	fetcherpb "clio/internal/fetcher_router/pocketbase"
	"clio/internal/link"
	"clio/internal/processor"

	"github.com/pocketbase/pocketbase/core"
)

// RegisterRouteFetcherPhase registers the route_fetcher phase.
func RegisterRouteFetcherPhase(orch *processor.Orchestrator, execs *processor.ExecutorRegistry, app core.App) {
	// Register phase definition
	orch.RegisterPhase(processor.PhaseDefinition{
		Name:         "route_fetcher",
		Dependencies: []string{},
	})

	// Create stores
	fetcherStore := fetcherpb.New(app)
	fetchedFileStore := fetchedfilepb.New(app)

	// Register executor
	execs.Register("route_fetcher", processor.ExecutorFunc(func(record processor.Record) *processor.ExecutorResult {
		l := record.(*link.Link)

		// Parse domain from initial_url
		parsedURL, err := url.Parse(l.InitialURL)
		if err != nil {
			return &processor.ExecutorResult{
				Error: err,
			}
		}

		domain := strings.ToLower(parsedURL.Host)
		// Remove www. prefix if present
		domain = strings.TrimPrefix(domain, "www.")

		// Look up fetcher configuration
		fr, err := fetcherStore.GetByDomain(domain)
		if err != nil {
			// If no specific route found, return error
			return &processor.ExecutorResult{
				Error: err,
			}
		}

		// Register fetcher phases dynamically
		fetcherPhases, err := fetcher.RegisterFetcherPhases(fr.Fetcher, orch, execs, fetchedFileStore)
		if err != nil {
			return &processor.ExecutorResult{
				Error: err,
			}
		}

		// Update orchestrator order to include fetcher phases
		newOrder := []string{"route_fetcher"}
		newOrder = append(newOrder, fetcherPhases...)
		orch.SetOrder(newOrder)

		// Save fetcher info and phases in phase results
		return &processor.ExecutorResult{
			Data: map[string]interface{}{
				"fetcher":        fr.Fetcher,
				"fetcher_config": fr.FetcherConfig,
				"domain":         domain,
				"phases":         fetcherPhases,
			},
		}
	}))
}
