package processor

// ExecutorResult contains the output of phase execution.
type ExecutorResult struct {
	Data  map[string]interface{}
	Error error
}

// Executor is the interface that phase workers must implement.
type Executor interface {
	// Execute performs the phase work for a Record
	Execute(record Record) *ExecutorResult
}

// ExecutorFunc is a simple function-based executor.
type ExecutorFunc func(record Record) *ExecutorResult

// Execute implements Executor interface.
func (f ExecutorFunc) Execute(record Record) *ExecutorResult {
	return f(record)
}

// ExecutorRegistry maps phase names to their executors.
type ExecutorRegistry struct {
	executors map[string]Executor
}

// NewExecutorRegistry creates a new executor registry.
func NewExecutorRegistry() *ExecutorRegistry {
	return &ExecutorRegistry{
		executors: make(map[string]Executor),
	}
}

// Register registers an executor for a phase.
func (r *ExecutorRegistry) Register(phaseName string, executor Executor) {
	r.executors[phaseName] = executor
}

// Get retrieves an executor by phase name.
func (r *ExecutorRegistry) Get(phaseName string) Executor {
	return r.executors[phaseName]
}
