package pocketbase

import (
	"clio/internal/fetched_file"

	"github.com/pocketbase/pocketbase/core"
	"github.com/pocketbase/pocketbase/tools/filesystem"
)

// Store manages FetchedFile records in PocketBase.
type Store struct {
	app core.App
}

// New creates a new PocketBase store for fetched_files.
func New(app core.App) *Store {
	return &Store{app: app}
}

// Get retrieves a fetched_file by ID from PocketBase.
func (s *Store) Get(id string) (*fetched_file.FetchedFile, error) {
	record, err := s.app.FindRecordById("fetched_files", id)
	if err != nil {
		return nil, err
	}

	created := record.GetDateTime("created")
	updated := record.GetDateTime("updated")

	ff := &fetched_file.FetchedFile{
		ID:      record.Id,
		LinkID:  record.GetString("link"),
		File:    record.GetString("file"),
		Created: created.Time(),
		Updated: updated.Time(),
	}

	return ff, nil
}

// GetByLink retrieves all fetched_files for a given link ID.
func (s *Store) GetByLink(linkID string) ([]*fetched_file.FetchedFile, error) {
	records, err := s.app.FindRecordsByFilter(
		"fetched_files",           // collection
		"link = {:linkID}",        // filter
		"-created",                // sort (newest first)
		0,                         // limit (0 = all)
		0,                         // offset
		map[string]any{"linkID": linkID}, // params
	)
	if err != nil {
		return nil, err
	}

	files := make([]*fetched_file.FetchedFile, len(records))
	for i, record := range records {
		created := record.GetDateTime("created")
		updated := record.GetDateTime("updated")

		files[i] = &fetched_file.FetchedFile{
			ID:      record.Id,
			LinkID:  record.GetString("link"),
			File:    record.GetString("file"),
			Created: created.Time(),
			Updated: updated.Time(),
		}
	}

	return files, nil
}

// Create creates a new fetched_file record with a file.
// fileBytes is the raw file content, filename is the desired filename.
// Returns the ID of the created record.
func (s *Store) Create(linkID string, fileBytes []byte, filename string) (string, error) {
	// Create PocketBase file from bytes
	file, err := filesystem.NewFileFromBytes(fileBytes, filename)
	if err != nil {
		return "", err
	}

	// Find fetched_files collection
	collection, err := s.app.FindCollectionByNameOrId("fetched_files")
	if err != nil {
		return "", err
	}

	// Create new record
	record := core.NewRecord(collection)
	record.Set("link", linkID)
	record.Set("file", file)

	// Save record
	if err := s.app.Save(record); err != nil {
		return "", err
	}

	return record.Id, nil
}

// Delete removes a fetched_file from PocketBase.
func (s *Store) Delete(id string) error {
	record, err := s.app.FindRecordById("fetched_files", id)
	if err != nil {
		return err
	}

	return s.app.Delete(record)
}
