package fetcher

import (
	"clio/internal/processor"
	"fmt"

	fetchedfilepb "clio/internal/fetched_file/pocketbase"
)

// RegisterFetcherPhases dynamically registers all phases for a fetcher.
// Returns the list of full namespaced phase names in execution order.
func RegisterFetcherPhases(
	fetcherName string,
	orch *processor.Orchestrator,
	execs *processor.ExecutorRegistry,
	store *fetchedfilepb.Store,
) ([]string, error) {
	// Get fetcher from registry
	f, err := Get(fetcherName)
	if err != nil {
		return nil, err
	}

	// Get phase definitions
	phases := f.Phases()
	if len(phases) == 0 {
		return nil, fmt.Errorf("fetcher %s has no phases", fetcherName)
	}

	fullPhaseNames := make([]string, len(phases))

	// Register each phase with namespacing
	for i, phaseDef := range phases {
		// Create full namespaced name
		fullName := fmt.Sprintf("fetch:%s:%s", fetcherName, phaseDef.Name)
		fullPhaseNames[i] = fullName

		// Map dependencies to full names
		fullDeps := make([]string, len(phaseDef.Dependencies))
		for j, dep := range phaseDef.Dependencies {
			fullDeps[j] = fmt.Sprintf("fetch:%s:%s", fetcherName, dep)
		}

		// Register phase definition with orchestrator
		err := orch.RegisterPhase(processor.PhaseDefinition{
			Name:         fullName,
			Dependencies: fullDeps,
		})
		if err != nil {
			return nil, fmt.Errorf("registering phase %s: %w", fullName, err)
		}

		// Get executor from fetcher
		executor, err := f.GetExecutor(phaseDef.Name, store)
		if err != nil {
			return nil, fmt.Errorf("getting executor for %s: %w", phaseDef.Name, err)
		}

		// Register executor
		execs.Register(fullName, executor)
	}

	return fullPhaseNames, nil
}
