"""
Authentication utilities for API key validation.
"""

import os
import secrets
import functools
from quart import request, jsonify

from dotenv import load_dotenv


def require_api_key(f):
    """
    Decorator to require API key in Authorization header.
    Expected format: Authorization: Bearer YOUR_API_KEY
    """
    @functools.wraps(f)
    async def decorated_function(*args, **kwargs):

        load_dotenv()

        auth_header = request.headers.get('Authorization')
        
        if not auth_header:
            return jsonify({"error": "Missing Authorization header"}), 401
        
        # Parse "Bearer {key}" format
        parts = auth_header.split()
        if len(parts) != 2 or parts[0].lower() != 'bearer':
            return jsonify({"error": "Invalid Authorization format. Use: Bearer YOUR_API_KEY"}), 401
        
        api_key = parts[1]
        
        # Get expected key from environment
        expected_key = os.getenv('PENELOPE_KEY')
        if not expected_key:
            return jsonify({"error": "Server not properly configured"}), 500
        
        # Validate key (use constant-time comparison to prevent timing attacks)
        if not secrets.compare_digest(api_key, expected_key):
            return jsonify({"error": "Invalid API key"}), 401
        
        # Key is valid, proceed
        return await f(*args, **kwargs)
    
    return decorated_function
