#!/usr/bin/env python3
"""
Press Enter Primitive Operation

This module provides realistic Enter key press functionality for browser automation.
Simulates human-like key press with natural timing.
"""

import asyncio
import random
from typing import Dict, Any
from patchright.async_api  import Page


class PressEnter:
    """
    Press Enter Key with Human-like Timing

    Implements realistic Enter key press behavior:
    - Small pre-press delay (hesitation)
    - Natural key press duration
    - Post-press delay for UI response

    Usage:
        presser = PressEnter()
        result = await presser.press(page, base_delay=50)
    """

    def __init__(self):
        """Initialize the press enter handler."""
        print("⏎ PressEnter initialized")

    async def press(self, page: Page, base_delay: int = 50) -> Dict[str, Any]:
        """
        Press the Enter key at the currently focused element.

        IMPORTANT: The target element should already be clicked/focused before calling this method.

        Args:
            page (Page): Playwright page to use
            base_delay (int): Base delay in milliseconds for timing (default: 50)

        Returns:
            Dict containing press result and timing information
        """
        try:
            print("⏎ Pressing Enter key")

            # Small pre-press hesitation (human doesn't immediately press Enter)
            pre_delay = base_delay * random.uniform(0.5, 1.5)
            await asyncio.sleep(pre_delay / 1000)

            # Press Enter key
            await page.keyboard.press('Enter')

            # Post-press delay (wait for UI to respond)
            post_delay = base_delay * random.uniform(1.0, 2.0)
            await asyncio.sleep(post_delay / 1000)

            result = {
                "success": True,
                "key": "Enter",
                "url": page.url
            }

            print("✅ Enter key pressed")
            return result

        except Exception as e:
            error_result = {
                "success": False,
                "error": str(e)
            }
            print(f"❌ Enter key press failed: {e}")
            return error_result
