#!/usr/bin/env python3
"""
Archive Save Plugin

Saves a URL to archive.is.
"""

import asyncio
from typing import Dict, Any
from lib.plugin_loader import PluginMetadata, PluginParameter
from lib.primitive.click_element import ClickElement
from lib.primitive.press_enter import PressEnter


class ArchiveSavePlugin:
    """
    Save a URL to archive.is.
    
    Supports two modes:
    - Normal: Navigate to archive.is, click URL input, submit
    - From check: Start from not-found page, click submit link
    """
    
    metadata = PluginMetadata(
        name="archive-save",
        description="Archive Save",
        methods=["POST"],
        bg_color=["#d3b71c", "#d8a80a"],
        parameters=[
            PluginParameter(
                name="url",
                type="url",
                label="URL to Save",
                required=False,
                placeholder="https://example.com",
                help_text="URL to save to archive.is (required unless from_check is true)"
            ),
            PluginParameter(
                name="from_check",
                type="checkbox",
                label="From check page",
                default=False,
                help_text="Start from not-found results page (skip navigation)"
            )
        ]
    )
    
    def validate_params(self, params: Dict[str, Any]) -> str | None:
        """Validate that URL is provided when not starting from check page."""
        url = params.get('url', '').strip()
        from_check = params.get('from_check', False)
        
        if not from_check and not url:
            return "URL is required when not starting from check page"
        
        return None
    
    async def execute(self, browser_navigator, page_id: str, params: Dict[str, Any]) -> Dict[str, Any]:
        """
        Execute archive save operation.
        
        Args:
            browser_navigator: BrowserNavigator instance
            page_id: Page to use for operation
            params: Validated parameters (url, from_check)
            
        Returns:
            Dict with success status and current URL
        """
        url = params.get('url', '').strip()
        from_check = params.get('from_check', False)
        
        print(f"💾 Archive save: url={url}, from_check={from_check}")
        
        managed_page = await browser_navigator._get_page_to_use(page_id)
        page = managed_page.page
        
        try:
            await managed_page.set_busy(f"saving {url} to archive")
            
            # Initialize primitives
            clicker = ClickElement()
            presser = PressEnter()
            
            if from_check:
                # Start from not found page - click the submit link
                print("💾 Starting from not found page, clicking submit link")
                submit_selector = '#CONTENT > div:nth-child(1) > div:nth-child(1) > ul:nth-child(2) > li:nth-child(1) > a:nth-child(1)'
                click_result = await clicker.click_selector(managed_page, submit_selector, slowmo=1000)
                
                if not click_result.get("success"):
                    return {
                        "success": False,
                        "url": url,
                        "error": f"Failed to click submit link: {click_result.get('error')}"
                    }
                
                # Press enter to submit
                print("💾 Pressing enter to submit")
                press_result = await presser.press(page, base_delay=50)
                
                if not press_result.get("success"):
                    return {
                        "success": False,
                        "url": url,
                        "error": f"Failed to press enter: {press_result.get('error')}"
                    }
                
                # Wait for page to process
                await asyncio.sleep(2)
                
            else:
                # Normal flow
                # Navigate to archive.is
                print("💾 Navigating to archive.is")
                await page.goto("https://archive.is/", wait_until="domcontentloaded")
                await asyncio.sleep(1)
                
                # Click URL input and press enter
                print("💾 Clicking URL input field")
                url_selector = '#url'
                click_result = await clicker.click_selector(managed_page, url_selector, slowmo=1000)
                
                if not click_result.get("success"):
                    return {
                        "success": False,
                        "url": url,
                        "error": f"Failed to click URL input: {click_result.get('error')}"
                    }
                
                # Press enter to submit
                print("💾 Pressing enter to submit")
                press_result = await presser.press(page, base_delay=50)
                
                if not press_result.get("success"):
                    return {
                        "success": False,
                        "url": url,
                        "error": f"Failed to press enter: {press_result.get('error')}"
                    }
                
                # Wait for submission to process
                await asyncio.sleep(2)
            
            result = {
                "success": True,
                "url": url if url else "(from not found page)",
                "from_check": from_check,
                "current_url": page.url,
                "message": f"Archive save initiated{' for ' + url if url else ' from not found page'}"
            }
            
            print(f"✅ Archive save completed")
            return result
        
        except Exception as e:
            print(f"❌ Archive save failed: {e}")
            await managed_page.set_error(str(e))
            raise
        
        finally:
            await managed_page.set_idle()
