#!/usr/bin/env python3
"""
Ops Router - Page management operations

This router handles page management operations like create, destroy, navigate, etc.
"""

import asyncio
import time
from quart import Blueprint, jsonify, request
from lib.page_management import PageState
from lib.auth import require_api_key



def register_ops_routes(url_api_verison, app, browser_navigator_getter):
    """
    Register ops routes with the app.

    Args:
        app: Quart app instance
        browser_navigator_getter: Function that returns the browser_navigator instance
    """

    ops = url_api_verison + "/ops"

    @app.route( ops + "/create-page", methods=["POST"])
    @require_api_key
    async def ops_create_page():
        """
        Create a new page/tab in the browser.

        Opens a new browser tab and returns page metadata with auto-generated or custom ID.

        Body parameters:
        - page_id: Custom page identifier (optional, auto-generated if not provided)
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized"}), 500

            data = await request.get_json() or {}
            page_id = data.get('page_id')

            result = await browser_navigator.create_page(page_id)

            if "error" in result:
                return jsonify(result), 400

            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route(ops + "/<page_id>/destroy-page", methods=["POST", "DELETE"])
    @require_api_key
    async def ops_destroy_page(page_id):
        """
        Destroy/close a page/tab in the browser.

        Closes the specified page tab and cleans up associated resources.

        URL parameters:
        - page_id: ID of the page to destroy
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized"}), 500

            result = await browser_navigator.destroy_page(page_id)

            if "error" in result:
                return jsonify(result), 400

            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.route(ops + "/<page_id>/navigate", methods=["POST"])
    @require_api_key
    async def ops_navigate(page_id):
        """
        Navigate to a URL in the specified page.

        Loads a URL in the page with automatic waiting for navigation to complete.
        Includes timeout protection to prevent indefinite hangs.

        URL parameters:
        - page_id: ID of the page to navigate in

        Body parameters:
        - url: Target URL to navigate to
        """
        request_id = f"api_{int(time.time() * 1000) % 10000}"

        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            await asyncio.sleep(2)

            data = await request.get_json()
            url = data.get('url')

            if not url:
                return jsonify({"error": "URL required", "page_id": page_id}), 400

            print(f"➡️ [{request_id}] API navigate request: {url} using page_id={page_id}")

            try:
                result = await asyncio.wait_for(
                    browser_navigator.navigate(url, page_id),
                    timeout=60.0
                )
                print(f"✅ [{request_id}] API navigate completed successfully")
                return jsonify(result)
            except asyncio.TimeoutError:
                print(f"⚠️ [{request_id}] API navigate timed out after 60 seconds")
                return jsonify({"error": "Navigation request timed out", "page_id": page_id}), 408

        except Exception as e:
            print(f"❌ [{request_id}] API navigate error: {e}")
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(ops + "/<page_id>/extract-content", methods=["POST"])
    @require_api_key
    async def ops_extract_content(page_id):
        """
        Extract full HTML content from the current page.

        Retrieves the complete DOM as HTML string for parsing and analysis.

        URL parameters:
        - page_id: ID of the page to extract content from
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            result = await browser_navigator.extract_content(page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.get(ops + "/<page_id>/screenshot")
    @require_api_key
    async def ops_screenshot(page_id):
        """
        Take a screenshot of the current page.

        Captures visual state of the page as an image file.

        URL parameters:
        - page_id: ID of the page to screenshot
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            result = await browser_navigator.screenshot(page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(ops + "/<page_id>/search-object", methods=["POST"])
    @require_api_key
    async def ops_search_object(page_id):
        """
        Search for an object on the page based on description using vision AI.

        Uses Gemini vision API to detect and locate objects matching the description.

        URL parameters:
        - page_id: ID of the page to search in

        Body parameters:
        - description: Description of the object to search for (required)
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json() or {}
            description = data.get('description')

            if not description:
                return jsonify({"error": "Object description required in request body", "page_id": page_id}), 400

            result = await browser_navigator.do_search_object(description, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500
