#!/usr/bin/env python3
"""
Primitive Router - Core browser operations

This router handles primitive browser operations like click, scroll, and type.
"""

from quart import jsonify, request
from lib.auth import require_api_key


def register_primitive_routes(url_api_verison, app, browser_navigator_getter):
    """
    Register primitive routes with the app.

    Args:
        app: Quart app instance
        browser_navigator_getter: Function that returns the browser_navigator instance
    """

    primitive = url_api_verison + "/primitive"

    @app.route(primitive + "/<page_id>/click-element", methods=["POST"])
    @require_api_key
    async def primitive_click_element(page_id):
        """Click an element on the page (primitive operation)."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json()
            selector = data.get('selector')

            if not selector:
                return jsonify({"error": "Selector required", "page_id": page_id}), 400

            result = await browser_navigator.click_element(selector, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/click-position", methods=["POST"])
    @require_api_key
    async def primitive_click_position(page_id):
        """Click at specific coordinates on the page (primitive operation)."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json()
            x = data.get('x')
            y = data.get('y')

            if x is None or y is None:
                return jsonify({"error": "X and Y coordinates required", "page_id": page_id}), 400

            result = await browser_navigator.click_position(x, y, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/scroll-page", methods=["POST"])
    @require_api_key
    async def primitive_scroll_page(page_id):
        """Scroll the page vertically (primitive operation)."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json() or {}
            distance = data.get('distance', 1000)

            result = await browser_navigator.scroll_page(distance, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/type-text", methods=["POST"])
    @require_api_key
    async def primitive_type_text(page_id):
        """Type text at the currently focused element (primitive operation).

        IMPORTANT: An element should already be clicked/focused before calling this endpoint.
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json()
            text = data.get('text')

            if not text:
                return jsonify({"error": "Text required", "page_id": page_id}), 400

            result = await browser_navigator.type_text(text, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/press-enter", methods=["POST"])
    @require_api_key
    async def primitive_press_enter(page_id):
        """Press the Enter key at the currently focused element (primitive operation).

        IMPORTANT: An element should already be clicked/focused before calling this endpoint.
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            result = await browser_navigator.press_enter(page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/press-tab", methods=["POST"])
    @require_api_key
    async def primitive_press_tab(page_id):
        """Press the Tab key to move focus to the next element (primitive operation).

        IMPORTANT: The page should be in a state where Tab navigation is expected.
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json() or {}
            with_shift = data.get('with_shift', False)

            result = await browser_navigator.press_tab(with_shift, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/paste-text", methods=["POST"])
    @require_api_key
    async def primitive_paste_text(page_id):
        """Paste text at the currently focused element using clipboard (primitive operation).

        IMPORTANT: An element should already be clicked/focused before calling this endpoint.
        This is faster than type-text for long content and mimics human copy-paste behavior.
        """
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json()
            text = data.get('text')

            if not text:
                return jsonify({"error": "Text required", "page_id": page_id}), 400

            result = await browser_navigator.paste_text(text, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/zoom-page", methods=["POST"])
    @require_api_key
    async def primitive_zoom_page(page_id):
        """Zoom the page at specified coordinates (primitive operation)."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            data = await request.get_json()
            x = data.get('x')
            y = data.get('y')
            zoom_level = data.get('zoom_level', 1)

            if x is None or y is None:
                return jsonify({"error": "X and Y coordinates required", "page_id": page_id}), 400

            result = await browser_navigator.zoom_page(x, y, zoom_level, page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500

    @app.route(primitive + "/<page_id>/reset-zoom", methods=["POST"])
    @require_api_key
    async def primitive_reset_zoom(page_id):
        """Reset page zoom to default level (primitive operation)."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized", "page_id": page_id}), 500

            result = await browser_navigator.reset_zoom(page_id)
            return jsonify(result)
        except Exception as e:
            return jsonify({"error": str(e), "page_id": page_id}), 500
