#!/usr/bin/env python3
"""
System Router - System and server management operations

This router handles system operations like ping, status, start, stop, and kill.
"""

import asyncio
import time
from quart import jsonify
from lib.page_management import PageState
from lib.auth import require_api_key



def register_system_routes(url_api_verison, app, browser_navigator_getter, browser_navigator_setter, shutdown_event, USER_DATA_DIR, SLOWMO, MAX_PAGES):
    """
    Register system routes with the app.

    Args:
        app: Quart app instance
        browser_navigator_getter: Function that returns the browser_navigator instance
        browser_navigator_setter: Function that sets the browser_navigator instance
        shutdown_event: Event for server shutdown
        USER_DATA_DIR: User data directory for browser
        SLOWMO: Slowmo delay
        MAX_PAGES: Maximum pages
    """

    system = url_api_verison  + "/system"

    @app.get(system + "/ping")
    async def system_ping():
        """Simple ping endpoint to test server responsiveness."""
        await asyncio.sleep(2)
        timestamp = int(time.time() * 1000)
        print(f"🏓 PING received at {timestamp}")
        return jsonify({"pong": timestamp, "status": "alive"})

    @app.get(system + "/status")
    async def system_status():
        """Get browser navigator status information."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized"}), 500

            # Get basic status from browser navigator
            page_pool = browser_navigator.get_page_pool()
            browser_context = browser_navigator.get_browser_context()

            status_info = {
                "initialized": True,
                "browser_context_active": browser_context is not None,
                "page_pool_active": page_pool is not None,
                "max_pages": browser_navigator.max_pages,
                "slowmo": browser_navigator.slowmo
            }

            if page_pool:
                status_info["page_count"] = len(page_pool.pages)
                status_info["available_pages"] = len([p for p in page_pool.pages.values() if p.state == PageState.IDLE])

            return jsonify(status_info)
        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.get(system + "/status/pages")
    async def system_status_pages():
        """Get detailed page pool status information."""
        try:
            browser_navigator = browser_navigator_getter()
            if not browser_navigator:
                return jsonify({"error": "Browser navigator not initialized"}), 500

            page_pool = browser_navigator.get_page_pool()
            if not page_pool:
                return jsonify({"error": "Page pool not initialized"}), 500

            pool_status = await page_pool.get_pool_status()
            return jsonify(pool_status)

        except Exception as e:
            return jsonify({"error": str(e)}), 500

    @app.get(system + "/start")
    @require_api_key
    async def system_start_browser():
        """Start/restart the browser navigator."""
        try:
            from lib.browser_navigator import BrowserNavigator

            browser_navigator = browser_navigator_getter()

            # Clean up existing navigator if it exists
            if browser_navigator:
                print("🔄 Shutting down existing browser navigator for restart...")
                try:
                    await browser_navigator.close()
                    print("✅ Existing browser navigator closed")
                except Exception as e:
                    print(f"⚠️ Error closing existing navigator: {e}")
                browser_navigator_setter(None)

            # Initialize new navigator
            print("🚀 Initializing new browser navigator...")
            browser_navigator = BrowserNavigator(
                user_data_dir=USER_DATA_DIR,
                slowmo=SLOWMO,
                max_pages=MAX_PAGES
            )
            await browser_navigator.initialize()
            browser_navigator_setter(browser_navigator)
            print("✅ Browser navigator restarted successfully")

            return jsonify({"success": True, "message": "Browser navigator started successfully"})
        except Exception as e:
            print(f"❌ Error starting browser navigator: {e}")
            return jsonify({"error": str(e)}), 500

    @app.get( system +  "/stop")
    @require_api_key
    async def system_stop_browser():
        """Stop the browser navigator without shutting down the server."""
        try:
            browser_navigator = browser_navigator_getter()

            if not browser_navigator:
                return jsonify({"success": True, "message": "Browser navigator is already stopped"})

            print("🛑 Stopping browser navigator...")
            try:
                await browser_navigator.close()
                print("✅ Browser navigator stopped successfully")
            except Exception as e:
                print(f"⚠️ Error stopping browser navigator: {e}")

            browser_navigator_setter(None)
            return jsonify({"success": True, "message": "Browser navigator stopped successfully"})
        except Exception as e:
            print(f"❌ Error stopping browser navigator: {e}")
            return jsonify({"error": str(e)}), 500

    @app.get(system + "/kill")
    @require_api_key
    async def system_kill_server():
        """Gracefully shutdown the server."""
        try:
            # Set the shutdown event to trigger cleanup
            shutdown_event.set()
            return jsonify({"success": True, "message": "Server shutdown initiated"})
        except Exception as e:
            return jsonify({"error": str(e)}), 500
